/*
  Background service worker: fetches offers from your hosted API to avoid CORS from content scripts.
*/

chrome.runtime.onMessage.addListener((msg, sender, sendResponse) => {
  if (!msg || msg.type !== 'SO_FETCH_OFFERS') return; // not ours

  const { merchant, country = 'FR' } = msg;
  const controller = new AbortController();
  const timeout = setTimeout(() => controller.abort(), 10000);

  (async () => {
    try {
      const u = new URL('https://student.flavienmathieu.fr/api/offers');
      if (merchant) u.searchParams.set('merchant', merchant);
      if (country) u.searchParams.set('country', country);

      const res = await fetch(u.toString(), {
        method: 'GET',
        headers: { 'Accept': 'application/json' },
        signal: controller.signal,
        // credentials cannot be included cross-origin in service worker fetch without CORS,
        // assume your API is public for reads or handles CORS. If cookies are needed
        // switch to "credentials: 'include'" and ensure server allows it.
      });

      if (!res.ok) {
        sendResponse({ ok: false, error: `HTTP ${res.status}` });
        return;
      }
      const json = await res.json().catch(() => ({ data: [] }));
      let items = Array.isArray(json)
        ? json
        : json.data || json.offers || json.results || json.items || [];
      if (!Array.isArray(items) && typeof items === 'object' && items !== null) {
        // handle shapes like { data: { items: [...] } }
        items = items.items || items.list || [];
      }

      if ((!items || items.length === 0) && typeof merchant === 'string' && merchant !== merchant.toLowerCase()) {
        try {
          const u2 = new URL('https://student.flavienmathieu.fr/api/offers');
          u2.searchParams.set('merchant', merchant.toLowerCase());
          if (country) u2.searchParams.set('country', country);
          const res2 = await fetch(u2.toString(), {
            method: 'GET',
            headers: { 'Accept': 'application/json' },
            signal: controller.signal,
          });
          if (res2.ok) {
            const json2 = await res2.json().catch(() => ({ data: [] }));
            let items2 = Array.isArray(json2)
              ? json2
              : json2.data || json2.offers || json2.results || json2.items || [];
            if (!Array.isArray(items2) && typeof items2 === 'object' && items2 !== null) {
              items2 = items2.items || items2.list || [];
            }
            items = Array.isArray(items2) ? items2 : [];
          }
        } catch (_) {}
      }

      sendResponse({ ok: true, data: Array.isArray(items) ? items : [] });
    } catch (e) {
      sendResponse({ ok: false, error: String(e && e.message || e) });
    } finally {
      clearTimeout(timeout);
    }
  })();

  // keep the message channel open for async response
  return true;
});
