(async () => {
  const mappingUrl = chrome.runtime.getURL("hosts.json");
  let hostMap = {};
  try {
    hostMap = await fetch(mappingUrl).then((r) => r.json());
  } catch (_) {}

  const normalizeHost = (h) => h.replace(/^www\./i, "").toLowerCase();
  const host = normalizeHost(location.hostname);

  const inferMerchant = (h) => {
    const base = h.split(".")[0] || h;
    return base
      .replace(/[-_]/g, " ")
      .replace(/\b\w/g, (c) => c.toUpperCase());
  };

  const merchant = hostMap[host] || inferMerchant(host);

  const requestOffers = (merchantValue) =>
    new Promise((resolve) => {
      chrome.runtime.sendMessage(
        {
          type: "SO_FETCH_OFFERS",
          merchant: typeof merchantValue === "string" ? merchantValue.toLowerCase() : merchantValue,
          country: "FR",
        },
        (resp) => {
          if (!resp || !resp.ok) return resolve([]);
          resolve(Array.isArray(resp.data) ? resp.data : []);
        }
      );
    });

  // Fetch first; only show UI if there are offers
  let offers = await requestOffers(merchant);
  offers = (offers || []).sort((a, b) => {
    const statusCmp = String(a.status || "").localeCompare(String(b.status || ""));
    if (statusCmp !== 0) return statusCmp;
    const aT = a.last_checked_at ? new Date(a.last_checked_at).getTime() : 0;
    const bT = b.last_checked_at ? new Date(b.last_checked_at).getTime() : 0;
    if (aT !== bT) return bT - aT;
    const aP = typeof a.discount_percent === "number" ? a.discount_percent : -Infinity;
    const bP = typeof b.discount_percent === "number" ? b.discount_percent : -Infinity;
    return bP - aP;
  });

  if (!offers || offers.length === 0) return; // no UI when no offers

  // Create FAB and overlay only when offers exist
  const fab = document.createElement("button");
  fab.id = "so-fab";
  fab.type = "button";
  fab.title = "Offres";
  fab.textContent = "Offres";
  document.documentElement.appendChild(fab);

  const overlay = document.createElement("div");
  overlay.id = "so-overlay";
  overlay.innerHTML = `
    <div class="so-header">
      <div class="so-title">Offres: <span class="so-merchant"></span></div>
      <button class="so-close" type="button" aria-label="Fermer">✕</button>
    </div>
    <div class="so-filters"></div>
    <div class="so-list"></div>
    <div class="so-empty" style="display:none;">Aucune offre trouvée.</div>
  `;
  document.documentElement.appendChild(overlay);

  overlay.querySelector(".so-merchant").textContent = merchant;
  const filtersEl = overlay.querySelector(".so-filters");
  const listEl = overlay.querySelector(".so-list");
  const emptyEl = overlay.querySelector(".so-empty");

  let activePlatforms = new Set();

  const renderFilters = () => {
    filtersEl.innerHTML = "";
    const platforms = Array.from(new Set(offers.map((o) => o.platform).filter(Boolean))).sort();
    if (platforms.length === 0) return;
    platforms.forEach((p) => {
      const pill = document.createElement("button");
      pill.type = "button";
      pill.className = "so-pill";
      pill.textContent = p;
      const activate = () => {
        if (activePlatforms.has(p)) {
          activePlatforms.delete(p);
          pill.classList.remove("active");
        } else {
          activePlatforms.add(p);
          pill.classList.add("active");
        }
        renderList();
      };
      pill.addEventListener("click", activate);
      filtersEl.appendChild(pill);
    });
    const clearBtn = document.createElement("button");
    clearBtn.type = "button";
    clearBtn.className = "so-clear";
    clearBtn.textContent = "Effacer";
    clearBtn.addEventListener("click", () => {
      activePlatforms.clear();
      filtersEl.querySelectorAll(".so-pill").forEach((el) => el.classList.remove("active"));
      renderList();
    });
    filtersEl.appendChild(clearBtn);
  };

  const renderList = () => {
    const filtered = offers.filter((o) => {
      if (activePlatforms.size === 0) return true;
      return activePlatforms.has(o.platform);
    });

    listEl.innerHTML = "";
    if (filtered.length === 0) {
      emptyEl.style.display = "block";
      return;
    }
    emptyEl.style.display = "none";

    filtered.forEach((o) => {
      const item = document.createElement("div");
      item.className = "so-item";
      const percent = o.discount_percent != null ? `${o.discount_percent}%` : "";
      const date = o.last_checked_at ? new Date(o.last_checked_at).toLocaleDateString("fr-FR") : "";
      const platform = o.platform || "";
      const country = o.country_code || o.country || "";
      const label = o.discount_label || "";

      item.innerHTML = `
        <div class="so-item-header">
          <div class="so-merchant-name">${o.merchant || merchant}</div>
          ${platform ? `<span class="so-badge">${platform}</span>` : ""}
        </div>
        <div class="so-item-body">
          ${percent ? `<span class="so-percent">${percent}</span>` : ""}
          ${label ? `<span class="so-label">${label}</span>` : ""}
        </div>
        <div class="so-item-footer">
          <span class="so-meta">${country ? country + " • " : ""}${date ? "vérifié le " + date : ""}</span>
          ${
            o.source_url
              ? `<a class="so-link" href="${o.source_url}" target="_blank" rel="noopener noreferrer">Source</a>`
              : ""
          }
        </div>
      `;
      listEl.appendChild(item);
    });
  };

  fab.addEventListener("click", () => {
    overlay.classList.toggle("open");
  });

  overlay.querySelector(".so-close").addEventListener("click", () => {
    overlay.classList.remove("open");
  });

  // Initial render
  renderFilters();
  renderList();
})();
